// Created on 15.03.2005
package biz.junginger.newsfeed;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.ImageLoader;
import org.eclipse.swt.graphics.Rectangle;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.InputSource;

import biz.junginger.newsfeed.eclipse.Logger;
import biz.junginger.newsfeed.model.ChannelContent;
import biz.junginger.newsfeed.net.CachedHttpContent;
import biz.junginger.newsfeed.net.MyFileCache;
import biz.junginger.newsfeed.net.UnexpectedHttpResponseCodeException;
import biz.junginger.newsfeed.parser.AtomFeedParser;
import biz.junginger.newsfeed.parser.FeedParser;
import biz.junginger.newsfeed.parser.RssFeedParser;

/**
 * Gets the content of an feed from the network or the cache, and calls an
 * appropriate parser (RSS or Atom). This class is thread-safe, e.g.
 * FeedProcessorQueue creates multiple threads operating on a single object.
 * 
 * (C) Copyright 2004-2008 Markus Junginger.
 * 
 * @author Markus Junginger
 */

/*
 * Copyright (c) 2009 Wind River Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License version
2.1 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the Lesser GNU General Public License version 2.1
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Contributor: 
 * Caroline Rieder (Wind River) - removed printStackTrace()
*/

/*
 * This file is part of RSS View.
 * 
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 * 
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class FeedProcessor
{
    private final FeedParser rssFeedParser;

    private final AtomFeedParser atomFeedParser;

    private DocumentBuilder docBuilder;

    private final MyFileCache cache;

    private final HashMap imageMap;

    public FeedProcessor(MyFileCache cache) 
    {
        this.cache = cache;
        rssFeedParser = new RssFeedParser();
        atomFeedParser = new AtomFeedParser();
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        try {
            docBuilder = factory.newDocumentBuilder();
        } catch (ParserConfigurationException ex) {
            throw new RuntimeException("Could not init FeedProcessor", ex); //$NON-NLS-1$
        }
        imageMap = new HashMap(); // TODO dispose images
    }

    private synchronized Image tryToGetFavIcon(URL fileUrl)
    {
        String protocol = fileUrl.getProtocol();
        String host = fileUrl.getHost();
        int port = fileUrl.getPort();

        try {
            URL url = new URL(protocol, host, port, "/favicon.ico"); //$NON-NLS-1$
            String key=url.toString();

            if (imageMap.containsKey(key)) {
                return (Image) imageMap.get(key);
            }

            CachedHttpContent content = new CachedHttpContent(cache, url);

            content.doRequest();
            byte[] bytes = content.getBytes();

            if (bytes == null) {
                imageMap.put(key, null);
                return null;
            }

            Image image = lookForIcon16(bytes);
            imageMap.put(key, image);
            return image;
        } catch (Exception ex) {
            if (ex.getMessage().indexOf("404") == -1) { //$NON-NLS-1$
                Logger.info("RSS view could not load favicon.ico for host " + host, ex); //$NON-NLS-1$
            }
            return null;
        }
    }

    private Image lookForIcon16(byte[] bytes) throws IOException
    {
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        ImageLoader loader = new ImageLoader();
        ImageData[] array = null;
        try {
            array = loader.load(in);
        } finally {
            in.close();
        }

        for (int i = 0; i < array.length; i++) {
            ImageData data = array[i];
            if (data.width == 16 && data.height == 16) {
                Image image = new Image(null, data);
                Rectangle rect = image.getBounds();
                if (rect.height == 16 && rect.width == 16)
                    return image;
            }
        }
        return null;
    }

    public synchronized ChannelContent processUrl(String urlStr) throws Exception
    {
    	URL url = new URL(urlStr);
    	CachedHttpContent httpContent = new CachedHttpContent(cache, url);
    	try{
    		httpContent.doRequest();
    	} catch(UnexpectedHttpResponseCodeException codeEx) {
    		Logger.warn("The provided RSS channel " + url + " does not exist.", codeEx); //$NON-NLS-1$ //$NON-NLS-2$
    	}
    	byte[] bytes = httpContent.getBytes();
    	if(bytes != null){
    		Document doc = createDocument(bytes);
    		Element root = doc.getDocumentElement();

    		ChannelContent content;
    		if ("feed".equals(root.getNodeName())) { //$NON-NLS-1$
    			content = atomFeedParser.parseDocument(doc);
    		} else {
    			content = rssFeedParser.parseDocument(doc);
    		}
    		content.setSourceUrl(urlStr);

    		Image favIcon = tryToGetFavIcon(url);
    		content.setIcon16(favIcon);
    		return content;
    	}
    	return null;
    }

    // synchronized, DocBuilder is not guarateed to be thread safe
    private synchronized Document createDocument(byte[] bytes) throws Exception
    {
        try {
            ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes);
            Document doc = docBuilder.parse(inputStream);
            return doc;
        } catch (Exception ex) {
            ByteArrayInputStream inputStream = new ByteArrayInputStream(bytes);
            InputSource source = new InputSource(inputStream);
            source.setEncoding("Cp-1252"); //$NON-NLS-1$
            try {
                return docBuilder.parse(source);
            } catch (Exception ex2) {
                throw ex;
            }
        }
    }

}
