// Created on 01.02.2008
package biz.junginger.newsfeed;

import java.net.NoRouteToHostException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import biz.junginger.newsfeed.eclipse.Logger;
import biz.junginger.newsfeed.model.ChannelContent;
import biz.junginger.newsfeed.model.ChannelSettings;

/**
 * Simple queue interface for a multithreading a FeedProcessor.
 * 
 * (C)Copyright 2004-2008 Markus Junginger.
 * 
 * @author Markus Junginger
 */
/*
 * This file is part of RSS View.
 * 
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 * 
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class FeedProcessorQueue
{
    private List resultQueue;

    private final FeedProcessor processor;

    private ExceptionHistory exceptionHistory;

    private List activeThreads;
    
    public static class Result{
        public final ChannelContent content;
        public final ChannelSettings settings;

        private Result(ChannelSettings settings, ChannelContent content)
        {
            this.settings = settings;
            this.content = content;
        }
    }

    public FeedProcessorQueue(FeedProcessor processor)
    {
        this.processor = processor;
        exceptionHistory = ExceptionHistory.getInstance();
        resultQueue = new ArrayList();
        activeThreads = new ArrayList();
    }

    public void submitWork(ChannelSettings settings)
    {
        ProcessorThread thread = new ProcessorThread(settings);
        thread.start();
        synchronized (activeThreads) {
            activeThreads.add(thread);
        }
    }

    public int getActiveThreadCount()
    {
        int count = 0;
        synchronized (activeThreads) {
            Iterator iterator = activeThreads.iterator();
            while (iterator.hasNext()) {
                ProcessorThread thread = (ProcessorThread) iterator.next();
                if (thread.isAlive()) {
                    count++;
                } else {
                    iterator.remove();
                }
            }
        }
        return count;
    }

    public boolean getThreadsActive()
    {
        return getActiveThreadCount() > 0;
    }

    /**
     * @return result or null
     */
    public Result dequeue()
    {
        synchronized (resultQueue) {
            if (resultQueue.isEmpty()) {
                return null;
            } else {
                return (Result) resultQueue.remove(0);
            }
        }
    }

    public void enqueue(Result result)
    {
        synchronized (resultQueue) {
            resultQueue.add(result);
        }
    }

    private class ProcessorThread extends Thread
    {
        private final ChannelSettings settings;

        private ProcessorThread(ChannelSettings settings)
        {
            this.settings = settings;
            setDaemon(true);
            setName("RSS View Feed ProcessorThread"); //$NON-NLS-1$
        }

        public void run()
        {
            String url = settings.getUrl();
            try {
                ChannelContent content = processor.processUrl(url);
                if(content != null){
                    Result result = new Result(settings, content);
                    enqueue(result);
                }
            } catch (NoRouteToHostException noroute) {
                exceptionHistory.add(noroute, "No route to " + url, settings); //$NON-NLS-1$
                settings.setError(noroute.getMessage(), noroute);
            } catch (UnknownHostException uhe) {
                exceptionHistory.add(uhe, "Unknown host: " + url, settings); //$NON-NLS-1$
                settings.setError(uhe.getMessage(), uhe);
            } catch (Exception ex) {
                exceptionHistory.add(ex, "Exception while processing " + url, settings); //$NON-NLS-1$
                if (settings.getException() == null) {
                    Logger.error("RSS view: problem with " + url, ex); //$NON-NLS-1$
                }
                settings.setError(ex.getMessage(), ex);
            }
        }
    }

}
