// Created on 18.03.2005
package biz.junginger.newsfeed;

import java.util.Iterator;
import java.util.List;

import biz.junginger.newsfeed.FeedProcessorQueue.Result;
import biz.junginger.newsfeed.eclipse.Logger;
import biz.junginger.newsfeed.model.ChannelModel;
import biz.junginger.newsfeed.model.ChannelSettings;
import biz.junginger.newsfeed.net.MyFileCache;

/**
 * This is the central controller thread, which checks if the feeds need
 * updates, trigger updates, and triggers updates.
 *
 * (C) Copyright 2004-2008 Markus Junginger.
 *
 * @author Markus Junginger
 */

/*
 * Copyright (c) 2009 Wind River Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License version
2.1 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the Lesser GNU General Public License version 2.1
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Contributor:
 * Caroline Rieder (Wind River) - changed printStackTrace() to logged error
*/

/*
 * This file is part of RSS View.
 *
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class FeedReaderThread extends Thread
{
    private int globalUpdateIntervallMinutes;

    private boolean active;

    private boolean shutdown;

    private FeedReaderThreadListener listener;

    private ChannelModel model;

    private FeedProcessorQueue feedProcessorQueue;

    private boolean threadsActive;

    public FeedReaderThread(ChannelModel model, FeedReaderThreadListener listener, MyFileCache cache)
    {
        this.model = model;
        this.listener = listener;
        globalUpdateIntervallMinutes = 5;
        FeedProcessor processor = new FeedProcessor(cache);
        feedProcessorQueue = new FeedProcessorQueue(processor);
        active = true;

        setName("RSS View Thread");
        setDaemon(true);
    }

    public int getGlobalUpdateIntervallMinutes()
    {
        return globalUpdateIntervallMinutes;
    }

    public void setGlobalUpdateIntervallMinutes(int intervall)
    {
        this.globalUpdateIntervallMinutes = intervall;
    }

    public void requestShutdown()
    {
        shutdown = true;
        try {
            interrupt();
        } catch (Throwable t) {
        }
    }

    @Override
    public void run()
    {
        while (!shutdown) {

            if (active) {
                try {
                    checkSubmitWork();
                    checkNotifyThreadListener();
                    checkUpdateModel();
                } catch (Throwable ex) {
                	Logger.error("An error occured.", ex); //$NON-NLS-1$
                    ExceptionHistory.getInstance().add(ex, getClass().getName());
                }
            }
            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
            }
        }
    }

    private void checkSubmitWork()
    {
        List copy = model.getSettingsCopy();
        if (copy != null) {
            Iterator it = copy.iterator();
            while (it.hasNext()) {
                ChannelSettings settings = (ChannelSettings) it.next();
                boolean update = (settings.isActive() && needsUpdate(settings));
                if (!shutdown && update) {
                    String url = settings.getUrl();
                    model.setLastUpdateMillis(url);
                    feedProcessorQueue.submitWork(settings);
                }
                if (shutdown) {
                    break;
                }
            }
        }
    }

    private boolean needsUpdate(ChannelSettings settings)
    {
        String url = settings.getUrl();
        if (model.getLastUpdateMillis(url) == 0) return true;

        long millis = System.currentTimeMillis() - model.getLastUpdateMillis(url);
        long age = millis / (1000 * 60);

        int maxAge = globalUpdateIntervallMinutes;
        int channelMinutes = settings.getUpdateIntervalMinutes();
        if (channelMinutes >= 1)
            maxAge = channelMinutes;

        return age > maxAge;
    }

    private void checkNotifyThreadListener()
    {
        boolean threadsActiveNew = feedProcessorQueue.getThreadsActive();
        if (threadsActiveNew && !threadsActive) {
            listener.feedReaderThreadStartsCheck();
        } else {
            if (!threadsActiveNew && threadsActive) {
                listener.feedReaderThreadEndsCheck();
            }
        }
        threadsActive = threadsActiveNew;
    }

    private void checkUpdateModel()
    {
        while (true) {
            Result result = feedProcessorQueue.dequeue();
            if (result == null) {
                break;
            }
            ChannelSettings settings = result.settings;
            String url = settings.getUrl();
            model.updateChannelContent(url, result.content);
            settings.clearError();
        }
    }


    public boolean isActive()
    {
        return active;
    }

    public void setActive(boolean active)
    {
        this.active = active;
    }

}
