package biz.junginger.newsfeed.eclipse;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import biz.junginger.newsfeed.ExceptionHistory;
import biz.junginger.newsfeed.ExceptionHistory.Entry;
import biz.junginger.newsfeed.model.ChannelSettings;

/**
 * (C) Copyright 2004-2008 Markus Junginger.
 * 
 * @author Markus Junginger
 */
/*
 * This file is part of RSS View.
 * 
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 * 
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class ExceptionHistoryDialog extends Dialog {

	private final ExceptionHistory history;
	private List urls;
	private StyledText textControl;
	private TreeViewer treeViewer;
	private DateFormat dateFormat;

	protected ExceptionHistoryDialog(Shell parentShell, ExceptionHistory history) {
		super(parentShell);
		this.history = history;
		Map urlStringMap = history.getUrlStringMapCopy();
		Set set = urlStringMap.keySet();
		urls = new ArrayList(set);
		dateFormat=new SimpleDateFormat("MM/dd HH:mm:ss");
	}

	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super.createDialogArea(parent);
		composite.setLayout(new GridLayout(2,false));
		
		Label label = new Label(composite, 0);
		label.setText("Errors by URL");

		Label label2 = new Label(composite, 0);
		label2.setText("Error detail (stack trace)");
		
		treeViewer = new TreeViewer(composite);
		treeViewer.setContentProvider(new MyContentProvider());
		treeViewer.setLabelProvider(new MyLabelProvider());
		treeViewer.setInput("dummy");
		treeViewer.addSelectionChangedListener(new ISelectionChangedListener() {
			public void selectionChanged(SelectionChangedEvent event) {
                IStructuredSelection selection= (IStructuredSelection) event.getSelection();
				Object element = selection.getFirstElement();
				updateText(element);
			}
		});
		GridData gdTree = new GridData();
		gdTree.widthHint=300;
		gdTree.heightHint=300;
		gdTree.grabExcessVerticalSpace=true;
		gdTree.verticalAlignment=SWT.BEGINNING;
		treeViewer.getTree().setLayoutData(gdTree);
		
		textControl = new StyledText(composite,0);
		textControl.setEditable(false);
		textControl.setWordWrap(false);
		GridData gd = new GridData();
		gd.heightHint=300;
		gd.widthHint=400;
		textControl.setLayoutData(gd);
		return composite;
	}

	private void updateText(Object source) {
		String text="";
		if(source instanceof ExceptionHistory.Entry) {
			ExceptionHistory.Entry entry=(Entry) source;
			if(entry.getOccured()!=null) {
				text+="Occured: "+dateFormat.format(entry.getOccured())+"\n\n";
			}
			if(entry.getAdditionalInfo()!=null) {
				text+=entry.getAdditionalInfo()+"\n\n";
			}
			Throwable th = entry.getThrowable();
			if(th!=null) {
				text+="Exception: "+th.getMessage()+"\n";
				StackTraceElement[] trace = th.getStackTrace();
				for (int i = 0; i < trace.length; i++) {
					text+="   "+trace[i].toString()+"\n";
				}
			}
		} else {
			text="Select an error entry in the tree on the left side";
		}
		textControl.setText(text);
	}
	protected void configureShell(Shell newShell) {
		super.configureShell(newShell);
		newShell.setText("Error History (RSS View)");
	}
	
    class MyContentProvider implements ITreeContentProvider
	{

		public Object[] getChildren(Object parent) {
			if(parent==null) {
				return urls.toArray();
			} else if (parent instanceof String) {
				List list = (List) history.getUrlStringMapCopy().get(parent);
				return list.toArray();
			}
			return null;
		}

		public Object getParent(Object element) {
			if(element instanceof ExceptionHistory.Entry) {
				ChannelSettings channelSettings = ((ExceptionHistory.Entry) element).getChannelSettings();
				if(channelSettings==null) {
					return ExceptionHistory.NO_URL_KEY;
				} else {
					return channelSettings.getUrl();
				}
			}
			return null;
		}

		public boolean hasChildren(Object element) {
			return getChildren(element)!=null;
		}

		public Object[] getElements(Object inputElement) {
			return getChildren(null);
		}

		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
    	
	}
	class MyLabelProvider extends LabelProvider implements ILabelProvider
	{
        public String getText(Object element) {
        	if(element instanceof String) {
        		return (String) element;
        	} else if(element instanceof ExceptionHistory.Entry) {
        		ExceptionHistory.Entry entry=(Entry) element;
        		String text="";
        		if(entry.getOccured()!=null) {
        			text+=dateFormat.format(entry.getOccured())+" ";
        		}
        		if(entry.getAdditionalInfo()!=null) {
        			text+=entry.getAdditionalInfo()+" ";
        		} else if(entry.getThrowable()!=null) {
        			text+=entry.getThrowable().getMessage()+" ";
        		}
        		return text;
        	}
            return "???";
        }
	}

}
