// Created on 25.02.2007
package biz.junginger.newsfeed.eclipse;

import java.net.MalformedURLException;

import org.eclipse.swt.SWT;
import org.eclipse.swt.browser.Browser;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackAdapter;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.FormText;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledFormText;

import biz.junginger.newsfeed.ExceptionHistory;
import biz.junginger.newsfeed.util.StringUtils;

/**
 * (C) Copyright 2004-2008 Markus Junginger.
 * 
 * @author Markus Junginger
 */

/*
 * Copyright (c) 2009 Wind River Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License version
2.1 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the Lesser GNU General Public License version 2.1
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Contributor: 
 * Caroline Rieder (Wind River) - Tooltip header now contains a "real" link, which can be used to open the feed
 * Caroline Rieder (Wind River) - Fixed tooltip rectangle position (focusing the tooltip was almost impossible on Windows) 
 * Caroline Rieder (Wind River) - substituted Browser widget for FormToolkit and ScrolledFormText
 * Caroline Rieder (Wind River) - changed printStackTrace() to logged error
 */

/*
 * This file is part of RSS View.
 * 
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 * 
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class FeedItemHover {
	/**
	 * Because Browser widget is not supported on Solaris yet, a Text widget is
	 * used to display the pop up content when the platform is Solaris.
	 * tagRepMap is a map which is used in formatting the content. It stores the
	 * what string should be used to replace a html tag.
	 */
	// TODO create property boolean textOnly
	static boolean isSunOS;
	static {
		String osName = System.getProperty("os.name").toLowerCase();
		isSunOS = osName.startsWith("sunos");
	}

	Shell hoverShell;

	private Table table;

	private Display display;

	private Browser browser;
	
	private FormToolkit toolkit;
	private FormText ftext;

	// If it is running on Solaris, a Text widget is used to display pop up
	// content.
	private Text textBrowser;

	private final ExceptionHistory exceptionHistory;

	private ViewItem lastOverViewItem;

	private TableItem lastOverTableItem;

	private boolean insideTable;

	private Link link;

	private Label titleLabel;

	private Label datePublishedLabel;

	private Label dateUpdatedLabel;

	private Composite headerComposite;

	private Image oldImage;

	private int additionalHeight;

	private int lastMouseX, lastMouseY;

	public FeedItemHover(Table table) {
		this.table = table;
		display = table.getDisplay();
		exceptionHistory = ExceptionHistory.getInstance();

		try {
			String osName = System.getProperty("os.name");
			if (osName.indexOf("Vista") != -1) {
				additionalHeight = 10;
			}

		} catch (Throwable th) {
			Logger.error("Property 'os.name' can not be found.", th); //$NON-NLS-1$
		}
		addListeners();
	}

	private void addListeners() {
		table.addMouseMoveListener(new MouseMoveListener() {
			public void mouseMove(MouseEvent e) {
				lastMouseX = e.x;
				lastMouseY = e.y;
				// TODO Broken fix: if (table.isFocusControl()) {
				processMouseMove(e.x, e.y);
				// }
			}

		});

		table.addFocusListener(new FocusListener() {

			public void focusGained(FocusEvent e) {
				// TODO check to show hover
				// processMouseMove(lastMouseX, lastMouseY);
			}

			public void focusLost(FocusEvent e) {
				// processMouseExits();
			}

		});

		table.addMouseTrackListener(new MouseTrackAdapter() {
			public void mouseExit(MouseEvent e) {
				insideTable = false;
				processMouseExits();
			}

			public void mouseEnter(MouseEvent e) {
				insideTable = true;
			}
		});

		table.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				dispose();
			}
		});
	}

	protected void processMouseMove(int x, int y) {
		try {
			Point p = new Point(x, y);
			TableItem item = table.getItem(p);
			ViewItem viewItem = null;
			if (item != null) {
				Object data = item.getData();
				if (data instanceof ViewItem) {
					viewItem = (ViewItem) data;
				}
			}
			processMouseOver(item, viewItem);
		} catch (RuntimeException ex) {
			Logger.error("Could not execute mouse move.", ex); //$NON-NLS-1$
			exceptionHistory.add(ex, "mouseMove");
		}

	}

	private void processMouseOver(final TableItem tableItem, final ViewItem viewItem) {
		if (viewItem == lastOverViewItem && lastOverTableItem == tableItem) {
			return;
		}

		lastOverViewItem = viewItem;
		lastOverTableItem = tableItem;

		display.timerExec(100, new Runnable() {

			public void run() {
				try {
					// Still over the same items?
							if (lastOverViewItem == viewItem && lastOverTableItem == tableItem) {
								if (viewItem == null || tableItem == null) {
									setVisible(false);
								} else {
									show(tableItem, viewItem);
								}
							}

				} catch (Throwable th) {
					Logger.error("Could not execute MouseOver", th); //$NON-NLS-1$
					exceptionHistory.add(th, "processMouseOver");
				}
			}
		});
	}

	private void processMouseExits() {
		display.timerExec(300, new Runnable() {
			public void run() {
				if (!insideTable && hoverShell != null && !hoverShell.isDisposed()) {
					Rectangle bounds = hoverShell.getBounds();
					final int tolerance = 20;
					Rectangle tolerantBounds = new Rectangle(bounds.x - tolerance, bounds.y - tolerance, bounds.width
							+ tolerance * 2, bounds.height + tolerance * 2);
					Point point = display.getCursorLocation();
					// System.out.println(point + " -> " + bounds + " -> " +
							// tolerantBounds);
					if (tolerantBounds.contains(point)) {
						processMouseExits();
					} else {
						setVisible(false);
					}
				} else {
					setVisible(false);
				}
			}
		});
	}

	public void setVisible(boolean visible) {
		if (hoverShell != null && !hoverShell.isDisposed()) {
			hoverShell.setVisible(visible);
		}
		if (!visible && browser != null && !browser.isDisposed()) {
			browser.stop();
		}
	}

	public void dispose() {
		if (hoverShell != null && !hoverShell.isDisposed()) {
			hoverShell.dispose();
		}
		hoverShell = null;
		browser = null;
		textBrowser = null;
		display = null;
		table = null;
	}

	public void show(TableItem tableItem, ViewItem viewItem) {
		if (hoverShell == null || hoverShell.isDisposed()) {
			createWidgets();
		}
		// TODO this is the wrong place to do this
		if (!table.isFocusControl()) {
			// Show hovering effect only when the table is the focus owner.
			// See https://rss-view.dev.java.net/issues/show_bug.cgi?id=4
			// TODO catch focus event and open hover immediatly
			return;
		}
		updateHeader(viewItem);

		String text = viewItem.getHtmlContent();
		if (text == null || text.length() == 0) {
			text = viewItem.getPlainTextContent();
			if (text == null || text.length() == 0) {
				text = "";
			}
		}
		boolean showHeaderOnly = false;
		if (textBrowser != null) {
			// If it is Solaris, format content and use Text to display the
			// content
			String plainText = viewItem.getPlainTextContent();
			textBrowser.setText(plainText);
			// Move the cursor to the beginning
			textBrowser.setSelection(0);
		}
		
		if(ftext != null){
			ftext.setText(viewItem.getPlainTextContent(), false, true);
		}	
		else if (browser != null) {
			browser.setText(text);
		}
		if (text.length() == 0) {
			showHeaderOnly = true;
		}

		Rectangle rectHoverShell = calcRect(tableItem, showHeaderOnly);
		hoverShell.setBounds(rectHoverShell);
		hoverShell.setVisible(true);
		hoverShell.moveAbove(null);
	}

	private Rectangle calcRect(TableItem tableItem, boolean showHeaderOnly) {
		Rectangle itemRect =  tableItem.getBounds(0);

		int sw = display.getBounds().width;
		int sh = display.getBounds().height;
		Point pt = table.toDisplay(itemRect.x, itemRect.y);

		int w = 450;
		int h = 300;
		if (showHeaderOnly) {
			int headerHeight = headerComposite.getSize().y;
			if (headerHeight > 0) {
				h = headerHeight + 5 + additionalHeight;
			}
		}

		int y = pt.y + itemRect.height / 2 - h / 2;
		y = Math.min(y, sh - h - 30);
		y = Math.max(y, 0);

		// Hovering shell should be right aligned with the second table column.
		// See https://rss-view.dev.java.net/issues/show_bug.cgi?id=3
		int tableWidth = table.getBounds().width;
		int columnsWidth = table.getColumn(0).getWidth() + table.getColumn(1).getWidth();
		int x = pt.x - 5 + Math.min(tableWidth, columnsWidth);

		int spaceRight = sw - x - w;
		if (spaceRight < 0) {
			int x2 = pt.x - w - 5;
			if (x2 < -100) {
				x = pt.x + 300;
			} else {
				x = x2;
			}
		}

		Rectangle rect = new Rectangle(x, y, w, h);
		return rect;
	}

	private void updateHeader(ViewItem viewItem) {
		String url = viewItem.getUrl();
		link.setText(url != null ? "<A>" + url.toString() + "</A>" : "No link available");
		String title = viewItem.getTitle();
		titleLabel.setText(title != null ? title : "No title");

		String published = "Published: ";
		String datePublished = viewItem.getDatePublished();
		if (datePublished != null && datePublished.length() > 0) {
			published += datePublished;
		} else {
			published += "unknown";
		}
		datePublishedLabel.setText(published);

		String updated = "Updated: ";
		String dateUpdated = viewItem.getDateUpdated();
		if (dateUpdated != null) {
			updated += dateUpdated;
		} else {
			updated += "unknown";
		}
		dateUpdatedLabel.setText(updated);
	}

	private void createWidgets() {
		hoverShell = new Shell(display, SWT.ON_TOP | SWT.TOOL | SWT.RESIZE | SWT.NO_FOCUS);
		Color white = display.getSystemColor(SWT.COLOR_WHITE);
		hoverShell.setBackground(white);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 1;
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		hoverShell.setLayout(gridLayout);
		createHeaderComposite();
		GridData data = new GridData();
		data.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		data.verticalAlignment = org.eclipse.swt.layout.GridData.FILL;
		data.horizontalIndent = 5;
		data.grabExcessVerticalSpace = true;
		data.grabExcessHorizontalSpace = true;
		
		toolkit = new FormToolkit(display);
		ScrolledFormText formText = new ScrolledFormText(hoverShell, SWT.H_SCROLL, false);
		formText.setLayoutData(data);
		formText.setExpandHorizontal(true);
		formText.setExpandVertical(true);
		formText.setBackground(white);
     	formText.addDisposeListener(new DisposeListener() {
     		public void widgetDisposed(DisposeEvent e) {
     			if (toolkit!=null) {
     				toolkit.dispose();
     				toolkit = null;
     			}
     		}
     	});
		ftext = toolkit.createFormText(formText, true);
		ftext.setHyperlinkSettings(toolkit.getHyperlinkGroup());
		ftext.addHyperlinkListener(new HyperlinkAdapter(){
			public void linkActivated(HyperlinkEvent e) {
				FeedPlugin plugin = FeedPlugin.getDefault();
				FeedPluginController controller = plugin.getController();
				try {
					String link = e.getHref().toString();
					if(link.endsWith(".")){
						link = link.substring(0, link.length()-1);
					}
					controller.getUrlOpener().openUrl(link);
				} catch (MalformedURLException e2) {
				} catch (Exception e3) {
				}
			}
		});
		formText.setFormText(ftext);
		
		if (browser == null && formText == null) {
			// Browser not available on Solaris and some Linux distros
			// Falback to Text. See also:
			// https://rss-view.dev.java.net/issues/show_bug.cgi?id=7
			// https://rss-view.dev.java.net/issues/show_bug.cgi?id=10
			textBrowser = new Text(hoverShell, SWT.MULTI | SWT.READ_ONLY | SWT.WRAP | SWT.V_SCROLL);
			textBrowser.setBackground(white);
			textBrowser.setLayoutData(data);
		}
	}

	/**
	 * This method initializes headerComposite
	 */
	private void createHeaderComposite() {
		GridLayout gridLayout = new GridLayout();
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.numColumns = 2;
		headerComposite = new Composite(hoverShell, SWT.NONE);
		headerComposite.setLayout(gridLayout);
		GridData compGD = new GridData();
		compGD.grabExcessHorizontalSpace = true;
		compGD.verticalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;
		compGD.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		headerComposite.setLayoutData(compGD);
		headerComposite.addControlListener(new ControlAdapter() {
			public void controlResized(ControlEvent e) {
				setHeaderImage();
			}
		});

		titleLabel = new Label(headerComposite, SWT.NONE | SWT.BOLD);
		setTitleLabelFont();

		GridData titleGD = new GridData();
		titleGD.verticalAlignment = org.eclipse.swt.layout.GridData.BEGINNING;
		titleGD.horizontalSpan = 2;
		titleGD.grabExcessHorizontalSpace = true;
		titleGD.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		titleLabel.setLayoutData(titleGD);

		link = new Link(headerComposite, SWT.NONE);
		GridData linkGD = new GridData();
		linkGD.grabExcessHorizontalSpace = true;
		linkGD.horizontalSpan = 2;
		linkGD.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		link.setLayoutData(linkGD);
		link.addListener(SWT.Selection, new Listener() {
			public void handleEvent(Event event) {
				FeedPlugin plugin = FeedPlugin.getDefault();
				FeedPluginController controller = plugin.getController();
				try {
					controller.getUrlOpener().openUrl(StringUtils.convertHtmlToPlainText(link.getText()));
				} catch (MalformedURLException e) {
				} catch (Exception e) {
				}
			}
		});

		GridData dateGD = new GridData();
		dateGD.grabExcessHorizontalSpace = true;
		dateGD.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
		datePublishedLabel = new Label(headerComposite, SWT.NONE);
		datePublishedLabel.setLayoutData(dateGD);
		dateUpdatedLabel = new Label(headerComposite, SWT.NONE);
		dateUpdatedLabel.setLayoutData(dateGD);

		setHeaderImage();
	}

	private void setTitleLabelFont() {
		boolean failed = false;
		try {
			FontData fd = new FontData("Tahoma", 9, SWT.BOLD);
			Font myFont = new Font(display, fd);
			titleLabel.setFont(myFont);
		} catch (Throwable th) {
			failed = true;
		}
		if (failed) {
			Font font = titleLabel.getFont();
			if (font != null) {
				FontData[] fontData = font.getFontData();
				if (fontData != null && fontData.length == 1) {
					fontData[0].setStyle(SWT.BOLD);
					Font boldFont = new Font(display, fontData);
					titleLabel.setFont(boldFont);
				}
			}
		}
	}

	protected void setHeaderImage() {
		Rectangle rect = headerComposite.getClientArea();
		Image newImage = new Image(display, 1, Math.max(1, rect.height));
		Color col1 = new Color(display, 0xdd, 0xdd, 0xdd);
		GC gc = new GC(newImage);
		gc.setForeground(col1);
		gc.setBackground(display.getSystemColor(SWT.COLOR_WHITE));
		gc.fillGradientRectangle(rect.x, rect.y, 1, rect.height, true);
		gc.dispose();
		col1.dispose();
		headerComposite.setBackgroundImage(newImage);
		headerComposite.setBackgroundMode(SWT.INHERIT_DEFAULT);
		if (oldImage != null) {
			oldImage.dispose();
		}
		oldImage = newImage;
	}

}
