// Created on 17.02.2006
package biz.junginger.newsfeed.eclipse;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;

import biz.junginger.newsfeed.BrowserDelegate;
import biz.junginger.newsfeed.ExceptionHistory;
import biz.junginger.newsfeed.FeedReaderThread;
import biz.junginger.newsfeed.FeedReaderThreadListener;
import biz.junginger.newsfeed.eclipse.prefs.FeedSettingsMemento;
import biz.junginger.newsfeed.eclipse.prefs.PrefsConstants;
import biz.junginger.newsfeed.model.Channel;
import biz.junginger.newsfeed.model.ChannelModel;
import biz.junginger.newsfeed.model.ChannelSettings;
import biz.junginger.newsfeed.model.IRSSChannelProvider;
import biz.junginger.newsfeed.net.MyFileCache;
import biz.junginger.plugin.common.UrlOpener;

/**
 * (C) Copyright 2004-2008 Markus Junginger.
 *
 * @author Markus Junginger
 */

/*
 * Copyright (c) 2009 Wind River Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License version
2.1 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the Lesser GNU General Public License version 2.1
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Contributor:
 * Caroline Rieder (Wind River) - View initialization now runs in a separate low priority job
 * Caroline Rieder (Wind River) - Added new function retrieveChannels to handle contributed Channels that come in via the new Extension Point
 * Caroline Rieder (Wind River) - New Extension Point: RSSChannelProvider
 * Caroline Rieder (Wind River) - changed printStackTrace() to logged error
 * Caroline Rieder (Wind River) - added missing refresh
 */

/*
 * This file is part of RSS View.
 *
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class FeedPluginController implements IPropertyChangeListener, FeedReaderThreadListener
{
	private FeedPlugin plugin;
	private ChannelModel model;
	private FeedReaderThread thread;
	private FeedView view;
	private MyFileCache cache;
	private Set channels;

	private boolean isInitComplete;
	private boolean isUpdating;
	private Model2ViewModel model2View;
	private BrowserDelegate browserDelegate;

	public FeedPluginController(FeedPlugin feedPlugin, ChannelModel channelModel) throws ParserConfigurationException {
		plugin = feedPlugin;
		model = channelModel;

		IPreferenceStore preferenceStore = plugin.getPreferenceStore();
		model2View = new Model2ViewModel(model,preferenceStore);
		propagateViewMode();
		initCache();
		thread=new FeedReaderThread(model,this, cache);
		initPrefs();

		Job job = new Job("Initializing Feeds..."){

			@Override
            protected IStatus run(IProgressMonitor monitor) {
				long start=System.currentTimeMillis();
				if(!monitor.isCanceled()) {
					refreshChannelSettings(monitor);
				}
				thread.start();
				long time = System.currentTimeMillis()-start;
				if(Logger.getDebugMode() > 0){
					Logger.info("RSS View Controller started in "+time+"ms");
				}
				return Status.OK_STATUS;
			}
		};
		job.setUser(true);
		job.setPriority(Job.LONG);
		job.schedule();

	}

	private void initCache()
	{
		File cacheRootDir = plugin.getBundleContext().getDataFile("./offline-feeds/");
		try {
			cache=new MyFileCache(cacheRootDir);
		} catch (IOException ex) {
			Logger.error("Cache creation failed.", ex);
			ExceptionHistory.getInstance().add(ex, "initCache");
		}
	}

	private void initPrefs()
	{
		IPreferenceStore preferenceStore = plugin.getPreferenceStore();
		preferenceStore.addPropertyChangeListener(this);
		boolean offline=preferenceStore.getBoolean(PrefsConstants.AVAILABLE_OFFLINE);
		cache.setActive(offline);

		boolean headlineInToolTip = preferenceStore.getBoolean(PrefsConstants.SHOW_HEADLINE_IN_TOOLTIP);
		model2View.setShowHeadlineInTooltip(headlineInToolTip);
	}

	public void registerAndInitView(FeedView view) {
		this.view = view;
		model2View.setView(view);
		view.setUpdateState(isUpdating);
		view.setViewItems(createInitalItems());
	}

	private ViewItem[] createInitalItems()
	{
		ViewItem[] items=new ViewItem[1];
		ViewItem item=new ViewItem();
		item.setTitle("No feeds available yet...");
		item.setPlainTextContent("If feeds are available, they are probably loading right now.\n" +
				"To add new feeds, use the preferences.\n\n" +
		"Double click this item to get some help.");
		item.setUrl("http://www.junginger.biz/eclipse/rss-view-documentation.html?from=RSS");
		items[0]=item;
		return items;
	}

	public void triggerUpdate()
	{
		if(model!=null) model.resetAllUpdateTimes();
	}

	@Override
    public void propertyChange(PropertyChangeEvent event)
	{
		String property = event.getProperty();
		Object value = event.getNewValue();
		if(property.equals(PrefsConstants.DEFAULT_UPDATE_INTERVAL))
		{
			if(value instanceof Integer)
			{
				int min=((Integer)value).intValue();
				thread.setGlobalUpdateIntervallMinutes(min);
			}
		}
		else if(property.equals(PrefsConstants.RSS_CHANNELS))
		{
			if(value instanceof String)
			{
				refreshChannelSettings(new NullProgressMonitor());
			}
		}
		else if(property.equals(PrefsConstants.AVAILABLE_OFFLINE))
		{
			if(value instanceof Boolean)
			{
				boolean offline = ((Boolean)value).booleanValue();
				cache.setActive(offline);
				if(!offline) {
					cache.clearAll();
				}
			}
		}
		else if(property.equals(PrefsConstants.SHOW_HEADLINE_IN_TOOLTIP))
		{
			if(value instanceof Boolean)
			{
				boolean on=((Boolean)value).booleanValue();
				model2View.setShowHeadlineInTooltip(on);
				model2View.updateView();
			}
		} else if(property.equals(PrefsConstants.VIEW_MODE)) {
			propagateViewMode();
		} else if(property.equals(PrefsConstants.FILTER_ACTIVE)) {
			propagateViewMode();
		} else if(property.equals(PrefsConstants.FILTER_ITEMS_PER_FEED)) {
			propagateViewMode();
		} else if(property.equals(PrefsConstants.FILTER_MINUTES)) {
			propagateViewMode();
		} else if(property.equals(PrefsConstants.FILTER_TIME_ACTIVE)) {
			propagateViewMode();
		} else if(property.equals(PrefsConstants.FILTER_TIME)) {
			propagateViewMode();
		}
	}


	/** @see biz.junginger.newsfeed.FeedReaderThreadListener#feedReaderThreadStartsCheck() */
	@Override
    public void feedReaderThreadStartsCheck()
	{
		isUpdating=true;
		if(view!=null) view.setUpdateState(true);
	}

	/** @see biz.junginger.newsfeed.FeedReaderThreadListener#feedReaderThreadEndsCheck() */
	@Override
    public void feedReaderThreadEndsCheck()
	{
		isInitComplete = true;
		isUpdating = false;
		if(view != null) view.setUpdateState(false);
	}

	public final boolean isInitComplete() {
		return isInitComplete;
	}

	public void refreshChannelSettings(IProgressMonitor monitor)
	{
		List list = null;
		IPreferenceStore store = plugin.getPreferenceStore();
		String str=store.getString(PrefsConstants.RSS_CHANNELS);
		if(str!=null) {
			try
			{
				list=FeedSettingsMemento.restoreFromXmlMementoString(str, monitor);
			} catch (Exception e)
			{
				Logger.error("XMLMementoString could not be restored.", e); //$NON-NLS-1$
				ExceptionHistory.getInstance().add(e, "refreshChannelSettings");
			}
		}
		if(channels == null){
			list = retrieveChannels(list);
		}
		try {
			str = FeedSettingsMemento.createXmlMementoString(list);
			store.setValue(PrefsConstants.RSS_CHANNELS, str);
			model.setChannelSettings(list);
		} catch (IOException e) {
			// CRI Auto-generated catch block
			e.printStackTrace();
		}
	}

	private List retrieveChannels(List listOfExistingChannels) {
		channels = new HashSet();
		IExtensionPoint extension = Platform.getExtensionRegistry().getExtensionPoint("biz.junginger.rss.eclipse.RssPlugin.RSSChannelProvider"); //$NON-NLS-1$
		if(extension != null) {
			IExtension[] extensions =  extension.getExtensions();
			for(int i = 0; i < extensions.length; i++) {
				IConfigurationElement[] elements = extensions[i].getConfigurationElements();
				for(int j = 0; j < elements.length; j++) {
					IConfigurationElement element = elements[j];
					if("RSSChannelProvider".equals(element.getName())) { //$NON-NLS-1$
						try {
							Object object = element.createExecutableExtension("class"); //$NON-NLS-1$
							if(object instanceof IRSSChannelProvider) {
								IRSSChannelProvider provider = (IRSSChannelProvider)object;
								Channel[] providedChannels = provider.provideChannels();
								if(providedChannels != null) {
									for(int k = 0; k < providedChannels.length; k++) {
										boolean found = false;
										for(int l = 0; l < listOfExistingChannels.size(); l++) {
											ChannelSettings setting = (ChannelSettings)listOfExistingChannels.get(l);
											if(setting.getUrl().equals(providedChannels[k].getUrl())) {
												found = true;
												break;
											}
										}
										if(!found) {
											channels.add(providedChannels[k]);
										}
									}
								}
							}
						} catch (InvalidRegistryObjectException e) {
							Logger.error("Invalid Registry Option.", e); //$NON-NLS-1$
						} catch (CoreException e) {
							Logger.error("Core Exception.", e); //$NON-NLS-1$
						}
					}
					else if("RSSChannelDefinition".equals(element.getName())) { //$NON-NLS-1$
							String url = element.getAttribute("url"); //$NON-NLS-1$
							if(url != null) {
								boolean found = false;
								for(int l = 0; l < listOfExistingChannels.size(); l++) {
									ChannelSettings setting = (ChannelSettings)listOfExistingChannels.get(l);
									if(setting.getUrl().equals(url)) {
										found = true;
										break;
									}
								}
								if(!found) {
									String updateIntervalMinutes = element.getAttribute("updateIntervalMinutes"); //$NON-NLS-1$
									String filterAgeMinutes = element.getAttribute("filterAgeMinutes"); //$NON-NLS-1$
									String isActive = element.getAttribute("isActive"); //$NON-NLS-1$
									boolean active = true;
									if(isActive.equals("true")){
										active = true;
									}
									else if(isActive.equals("false")){
										active = false;
									}
									channels.add(new Channel(url, Integer.parseInt(updateIntervalMinutes), Integer.parseInt(filterAgeMinutes), active));
								}
							}
					}
				}
			}
		}
		Set setOfProvidedChannels = new HashSet();
		for (Iterator iterator = channels.iterator(); iterator.hasNext();) {
			ChannelSettings setting = new ChannelSettings();
			Object x = iterator.next();
			if (x instanceof Channel){
				Channel channel = (Channel) x;
				setting.setUrl(channel.getUrl());
				setting.setActive(channel.isActive());
				setting.setFilterAgeMinutes(channel.getFilterAgeMinutes());
				setting.setUpdateIntervalMinutes(channel.getUpdateIntervalMinutes());
				setOfProvidedChannels.add(setting);
			}
		}
		listOfExistingChannels.addAll(setOfProvidedChannels);
		return listOfExistingChannels;
	}

	public synchronized void stop() {
		FeedReaderThread oldThread = thread;
		thread=null;
		oldThread.requestShutdown();
		try {
			oldThread.join(2000);
		} catch (InterruptedException ex) {
			Logger.error("Thread shutdown failed.", ex); //$NON-NLS-1$
		}
		model2View.dispose();
	}

	public void selectedViewModel(String mode) {
		IPreferenceStore prefs = plugin.getPreferenceStore();
		prefs.setValue(PrefsConstants.VIEW_MODE,mode);
	}

	public void propagateViewMode()
	{
		IPreferenceStore prefs = plugin.getPreferenceStore();
		String mode = prefs.getString(PrefsConstants.VIEW_MODE);
		if(PrefsConstants.VALUE_VIEW_MODE_GROUPED_BY_FEED.equals(mode)) {
			model2View.setMode(Model2ViewModel.MODE_GROUPED_BY_FEED);
		} else if(PrefsConstants.VALUE_VIEW_MODE_GROUPED_BY_TIME.equals(mode)) {
			model2View.setMode(Model2ViewModel.MODE_GROUPED_BY_TIME);
		} else if(PrefsConstants.VALUE_VIEW_MODE_GROUPED_BY_FEED_KEEP_SEQ.equals(mode)) {
			model2View.setMode(Model2ViewModel.MODE_GROUPED_BY_FEED_KEEP_SEQ);
		} else {
			model2View.setMode(Model2ViewModel.MODE_TIME_SORTED);
		}
		model2View.updateView();
	}

	public void expandOrCollapse(ViewItem item)
	{
		model2View.expandOrCollapse(item);
		model2View.updateView();
	}

	public UrlOpener getUrlOpener() {
		if(browserDelegate==null) {
			browserDelegate = new BrowserDelegate(plugin.getPreferenceStore());
		}
		return browserDelegate;
	}

	public FeedView getView()
	{
		return view;
	}

}