// Created on 18.03.2005
package biz.junginger.newsfeed.eclipse;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.MouseTrackListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IViewSite;

import biz.junginger.newsfeed.ExceptionHistory;

/**
 * (C) Copyright 2004-2008 Markus Junginger.
 * 
 * @author Markus Junginger
 */

/*
 * Copyright (c) 2009 Wind River Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License version
2.1 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the Lesser GNU General Public License version 2.1
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Contributor: 
 * Caroline Rieder (Wind River) - changed printStackTrace() to logged error
 */

/*
 * This file is part of RSS View.
 * 
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 * 
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class FeedTable
{
    private TableViewer viewer;

    // private ChannelItem[] rssItems;

    private ViewItem lastOverViewItem;

    private TableItem lastOverTableItem;

    private Color overCellColor = new Color(null, 220, 220, 255);

    private TableColumn textColumn;

    private ViewItem[] viewItems;

    private Font normalFont;

    private Font boldFont;

    protected boolean disposed;

    private Image expandImage;

    private Image collapseImage;

    private final FeedPluginController controller;

    private ExceptionHistory exceptionHistory;

    private FeedItemHover feedItemHover;

    public FeedTable(Composite parent, IViewSite site, FeedPluginController controller)
    {
        this.controller = controller;
        exceptionHistory = ExceptionHistory.getInstance();
        viewer = new TableViewer(parent, SWT.FILL | SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
        initFonts();
        final Table table = viewer.getTable();
        int columnFlags = SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION;
        TableColumn column = new TableColumn(table, columnFlags);
        column.setWidth(20);
        column.setResizable(false);

        column = new TableColumn(table, columnFlags);
        textColumn = column;

        viewer.setContentProvider(new MyContentProvider());
        viewer.setLabelProvider(new MyLabelProvider());
        viewer.setInput("Dummy, content is injected");

        ImageBank imageBank = ImageBank.getInstance();
        expandImage = imageBank.createImage(ImageBank.EXPAND);
        collapseImage = imageBank.createImage(ImageBank.COLLAPSE);

        addListeners();

        feedItemHover = new FeedItemHover(viewer.getTable());
    }

    private void initFonts()
    {
        final Table table = viewer.getTable();
        Display display = table.getDisplay();
        try {
            FontData[] fontData = table.getFont().getFontData();
            if (fontData != null && fontData.length > 0 && fontData[0] != null) {
                FontData fd = fontData[0];
                boldFont = new Font(display, fd.getName(), fd.getHeight(), fd.getStyle() | SWT.BOLD);
                normalFont = new Font(display, fd.getName(), fd.getHeight(), fd.getStyle());
            }
        } catch (Throwable ex) {
        	Logger.error("Font initialization failed.", ex); //$NON-NLS-1$
            exceptionHistory.add(ex, "initFonts");
        }
    }

    /**
     * @param table
     */
    private void addListeners()
    {
        /*
         * viewer.addDoubleClickListener(new IDoubleClickListener() { public
         * void doubleClick(DoubleClickEvent event) { try { ISelection selection =
         * viewer.getSelection(); Object obj =
         * ((IStructuredSelection)selection).getFirstElement(); if(obj
         * instanceof ViewItem) { onDoubleclick((ViewItem) obj); } }
         * catch(RuntimeException ex) { ex.printStackTrace(); } } });
         */

        final Table table = viewer.getTable();
        table.addMouseListener(new MouseListener()
        {
            public void mouseDoubleClick(MouseEvent e)
            {
                processMouseClick(e, true);
            }

            public void mouseDown(MouseEvent e)
            {
                if (e.button == 1) {
                    processMouseClick(e, false);
                }
            }

            public void mouseUp(MouseEvent e)
            {
            }

        });
        table.addMouseMoveListener(new MouseMoveListener()
        {
            public void mouseMove(MouseEvent e)
            {
                try {
                    Point p = new Point(e.x, e.y);
                    TableItem item = table.getItem(p);
                    if (item == null) {
                        onOverViewItem(null, null);
                        return;
                    }
                    Object data = item.getData();
                    if (data instanceof ViewItem) {
                        ViewItem viewItem = (ViewItem) data;
                        onOverViewItem(item, viewItem);
                    }
                } catch (RuntimeException ex) {
                	Logger.error("Mouse move failed.", ex); //$NON-NLS-1$
                    exceptionHistory.add(ex, "mouseMove");
                }
            }

        });

        table.addDisposeListener(new DisposeListener()
        {
            public void widgetDisposed(DisposeEvent e)
            {
                dispose();
            }
        });
        
        MouseTrackListener track = new MouseTrackListener()
        {

            public void mouseEnter(MouseEvent e)
            {
            }

            public void mouseExit(MouseEvent e)
            {
                onOverViewItem(null, null);
            }

            public void mouseHover(MouseEvent e)
            {
            }
        };
        table.addMouseTrackListener(track);
    }

    protected void processMouseClick(MouseEvent e, boolean doubleClick)
    {
        try {
            ISelection selection = viewer.getSelection();
            Object obj = ((IStructuredSelection) selection).getFirstElement();
            if (obj instanceof ViewItem) {
                if (doubleClick) {
                    onDoubleclick((ViewItem) obj);
                } else {
                    onSingleclick((ViewItem) obj);
                }
            }
        } catch (RuntimeException ex) {
        	Logger.error("Mouse click failed.", ex); //$NON-NLS-1$
            exceptionHistory.add(ex, "mouseClick");
        }

    }

    protected void dispose()
    {
        disposed = true;
        if(feedItemHover!=null) {
            feedItemHover.dispose();
            feedItemHover=null;
        }
        if (normalFont != null) {
            normalFont.dispose();
            normalFont = null;
        }
        if (boldFont != null) {
            boldFont.dispose();
            boldFont = null;
        }
    }

    public synchronized void setViewItems(ViewItem[] viewItems)
    {
        if (disposed)
            return;
        this.viewItems = viewItems;
        Runnable uiUpdater = new Runnable()
        {
            public void run()
            {
                if (disposed)
                    return;
                synchronized (FeedTable.this) {
                    viewer.refresh();
                    setTableItemFonts();
                    textColumn.pack();
                }
            }

            private void setTableItemFonts()
            {
                try {
                    TableItem[] items = viewer.getTable().getItems();
                    for (int i = 0; i < items.length; i++) {
                        TableItem item = items[i];
                        Object data = item.getData();
                        if (data instanceof ViewItem) {
                            boolean bold = ((ViewItem) data).isBoldText();
                            if (bold)
                                item.setFont(boldFont);
                            else
                                item.setFont(normalFont);
                        }
                    }
                } catch (RuntimeException ex) {
                	Logger.error("Setting table item fonts failed.", ex); //$NON-NLS-1$
                    exceptionHistory.add(ex, "setTableItemFonts");
                }
            }
        };
        Table table = viewer.getTable();
        if (!table.isDisposed()) {
            table.getDisplay().asyncExec(uiUpdater);
        }
    }

    /**
     * @param item
     * @param viewItem
     */
    protected void onOverViewItem(TableItem tableItem, ViewItem viewItem)
    {
        if (viewItem == lastOverViewItem && lastOverTableItem == tableItem) {
            return;
        }
        if (lastOverTableItem != null && !lastOverTableItem.isDisposed()) {
            setTableItemHighlight(lastOverTableItem, false);
            if (lastOverViewItem.isCollapsable()) {
                lastOverTableItem.setImage(lastOverViewItem.getImageLevel1());
            }
        }

        lastOverViewItem = viewItem;
        lastOverTableItem = tableItem;

        if (viewItem != null && viewItem.isCollapsable()) {
            Image image;
            if (viewItem.isCollapsed()) {
                image = expandImage;
            } else {
                image = collapseImage;
            }
            if (!tableItem.isDisposed()) {
                tableItem.setImage(image);
            }
        }
        if(tableItem!=null) {
            setTableItemHighlight(tableItem, true);
        }
    }

    private void setTableItemHighlight(TableItem tableItem, boolean highlight)
    {
        if (!tableItem.isDisposed()) {
            Color color = highlight ? overCellColor : null;
            tableItem.setBackground(1, color);
        }
    }

    protected void onSingleclick(ViewItem item)
    {
        if (item.isCollapsable()) {
            controller.expandOrCollapse(item);
        }
    }

    protected void onDoubleclick(ViewItem item)
    {
        final String link = item.getUrl();
        if (link == null) {
            showErrorMessage("No link was detected for this feed item");
            return;
        }

        Runnable linkThread = new Runnable()
        {
            public void run()
            {
                try {
                    controller.getUrlOpener().openUrl(link);
                } catch (Exception ex) {
                	Logger.error("Launching browser for " + link + " failed.", ex); //$NON-NLS-1$
                    showErrorMessage("Could not launch browser for URL '" + link + "'");
                }
            }
        };
        new Thread(linkThread).start();
    }

    private void showErrorMessage(String msg)
    {
        FeedView view = FeedPlugin.getDefault().getController().getView();
        if (view != null)
            view.openErrorDialog("RSS feed error", msg);
        else
            Logger.error(msg);
    }

    class MyContentProvider implements IStructuredContentProvider
    {
        public void inputChanged(Viewer v, Object oldInput, Object newInput)
        {
        }

        public void dispose()
        {
        }

        public Object[] getElements(Object parent)
        {
            if (viewItems == null)
                return new Object[0];
            Object[] o = new Object[viewItems.length];
            for (int i = 0; i < o.length; i++) {
                o[i] = viewItems[i];
            }
            return o;
        }
    }

    static class MyLabelProvider extends LabelProvider implements ITableLabelProvider
    {
        public MyLabelProvider()
        {
        }

        public String getColumnText(Object obj, int index)
        {
            if (index == 0)
                return "";
            if (obj instanceof ViewItem) {
                ViewItem item = (ViewItem) obj;
                if (index == 1) {
                    return item.getTitle();
                }
                return "";
            }
            return "Invalid";
        }

        public Image getColumnImage(Object obj, int index)
        {
            if (obj instanceof ViewItem) {
                ViewItem item = (ViewItem) obj;
                if (index == 0)
                    return item.getImageLevel1();
                if (index == 1)
                    return item.getImageLevel2();
            }
            return null;
        }

    }

    public TableViewer getViewer()
    {
        return viewer;
    }

}
