package biz.junginger.newsfeed.eclipse;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.IInputValidator;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.part.ViewPart;

import biz.junginger.newsfeed.ExceptionHistory;
import biz.junginger.newsfeed.eclipse.prefs.FeedSettingsMemento;
import biz.junginger.newsfeed.eclipse.prefs.PrefsConstants;
import biz.junginger.newsfeed.model.ChannelSettings;
import biz.junginger.plugin.common.PluginUtils;


/**
 * (C) Copyright 2004-2008 Markus Junginger.
 *
 * @author Markus Junginger
 */

/*
 * Copyright (c) 2009 Wind River Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License version
2.1 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the Lesser GNU General Public License version 2.1
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Contributor:
 * Caroline Rieder (Wind River) - ContextMenu: renamed actions to follow the Capitalization Guidelines
 * Caroline Rieder (Wind River) - Fixed Reload action (did not really refresh, but only the timers have been reset)
 * Caroline Rieder (Wind River) - ContextMenu: added new action: Open Feed
 * Caroline Rieder (Wind River) - changed printStackTrace() to logged error
 */

/*
 * This file is part of RSS View.
 *
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 *
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class FeedView extends ViewPart
{
    private final static String URL_PLUGIN_HOME = "http://www.junginger.biz/eclipse?from=RSS";

    private final static String URL_OPEN_SOURCE_HOME = "https://rss-view.dev.java.net/";

    private final static String URL_PLUGIN_DOC = "http://www.junginger.biz/eclipse/rss-view-documentation.html?from=RSS";

    private final static String PLUGIN_ID_INFO = "1001";

    private final static String PLUGIN_ID_CENTRAL = "369";

    public final static String VIEW_ID = "biz.junginger.newsfeed.eclipse.FeedView";

    private Composite parent;

    protected boolean visible;



    private Action reloadAction;

    private Action openFeedAction;



    private FeedTable feedTable;

    private FeedPluginController controller;

    private IPreferenceStore preferenceStore;

    private Image imageReload;

    private Image imageTitle;

    private ViewModeAction viewModeAction;

    private FilterAction filterAction;

    private IAction prefAction;

    private Action addAction;



    private Action showErrorsAction;

    private Image imageTitleNoNews;

    private int displayedItemCount;

    private int availableItemCount;

    private boolean updating;

    private Image currentTitleImage;

    public FeedView()
    {
        ImageBank imageBank = ImageBank.getInstance();
        imageReload = imageBank.createImage(ImageBank.RELOAD);
        imageTitle = imageBank.createImage(ImageBank.TITLE);
        imageTitleNoNews = imageBank.createImage(ImageBank.TITLE_GRAY);
    }

    @Override
    public void dispose()
    {
        super.dispose();
        if (imageReload != null) {
            imageReload.dispose();
            imageReload = null;
        }
        if (imageTitle != null) {
            imageTitle.dispose();
            imageTitle = null;
        }
        if (imageTitleNoNews != null) {
            imageTitleNoNews.dispose();
            imageTitleNoNews = null;
        }
    }

    @Override
    public void createPartControl(Composite parent)
    {
        this.parent = parent;
        FeedPlugin plugin = FeedPlugin.getDefault();
        controller = plugin.getController();

        feedTable = new FeedTable(parent, getViewSite(), controller);

        preferenceStore = plugin.getPreferenceStore();
        createActions();
        contributeToActionBars();
        createContextMenu();

        controller.registerAndInitView(this);
        controller.triggerUpdate();
    }

    public void setViewItems(ViewItem[] viewItems)
    {
        if (feedTable != null)
            feedTable.setViewItems(viewItems);
    }

    public FeedTable getFeedTable()
    {
    	return feedTable;
    }

    private void contributeToActionBars()
    {
        IActionBars bars = getViewSite().getActionBars();

        bars.setGlobalActionHandler(ActionFactory.REFRESH.getId(), reloadAction);
        IToolBarManager toolBarManager = bars.getToolBarManager();

        toolBarManager.add(createSetTimeFilterToNowAction());
        toolBarManager.add(filterAction);
        toolBarManager.add(viewModeAction);
        IMenuManager manager = bars.getMenuManager();

        addStandardActions(manager);
    }

    private Action createSetTimeFilterToNowAction()
    {
        Action setTimeFilterToNowAction = new Action()
        {
            @Override
            public void run()
            {
                filterAction.setTimeFilterToNow();
            }
        };
        ImageDescriptor imgDesc = ImageBank.getInstance().getImageDescriptor(ImageBank.TIME_FILTER);
        setTimeFilterToNowAction.setImageDescriptor(imgDesc);
        String tooltip = "Set time filter to current time\nHides all items until new items are available";
        setTimeFilterToNowAction.setToolTipText(tooltip);
        return setTimeFilterToNowAction;
    }

    private void addStandardActions(IMenuManager manager)
    {

        manager.add(openFeedAction);
        manager.add(addAction);
        if (prefAction != null) {
            manager.add(prefAction);
        }
        manager.add(new Separator());
        manager.add(reloadAction);
        manager.add(new Separator());
        manager.add(showErrorsAction);
    }

    private void createContextMenu()
    {
        // Create menu manager.
        MenuManager menuMgr = new MenuManager();
        menuMgr.setRemoveAllWhenShown(true);
        menuMgr.addMenuListener(new IMenuListener()
        {
            @Override
            public void menuAboutToShow(IMenuManager mgr)
            {
                mgr.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
                addStandardActions(mgr);
            }
        });

        // Create menu.
        TableViewer viewer = feedTable.getViewer();
        Menu menu = menuMgr.createContextMenu(viewer.getControl());
        viewer.getControl().setMenu(menu);

        // Register menu for extension.
        getSite().registerContextMenu(menuMgr, viewer);
    }

    protected void addFeedUrl(String url)
    {
        String string = preferenceStore.getString(PrefsConstants.RSS_CHANNELS);
        try {
            List list;
            if (string != null) {
                list = FeedSettingsMemento.restoreFromXmlMementoString(string, new NullProgressMonitor());
            } else {
                list = new ArrayList();
            }
            ChannelSettings settings = new ChannelSettings();
            settings.setActive(true);
            settings.setUrl(url);
            list.add(settings);
            String newMemento = FeedSettingsMemento.createXmlMementoString(list);
            preferenceStore.setValue(PrefsConstants.RSS_CHANNELS, newMemento);
        } catch (Exception ex) {
        	Logger.error("Could not add new feed.", ex); //$NON-NLS-1$
            ExceptionHistory.getInstance().add(ex, "addFeedUrl");
            MessageDialog.openError(null, "Could not add new feed", ex.toString());
        }

    }

    public boolean isVisible()
    {
        Runnable runDialog = new Runnable()
        {
            @Override
            public void run()
            {
                visible = parent.isVisible();
            }
        };

        Display display = getDisplay();
        if (display == null) {
            return false;
        } else {
            display.syncExec(runDialog);
            return visible;
        }
    }

    private Display getDisplay()
    {
        Shell widget = getViewSite().getShell();
        if (widget != null && !widget.isDisposed()) {
            return widget.getDisplay();
        } else {
            return null;
        }
    }

    private void createActions()
    {






        filterAction = new FilterAction(parent, preferenceStore);
        viewModeAction = new ViewModeAction(parent, preferenceStore);
        prefAction = PluginUtils.createPreferencesAction("biz.junginger.newsfeed.eclipse.FeedPreferencePage");

        createReloadAction();
        createOpenFeedAction();
        createAddAction();

        createShowErrorsAction();
    }

    private void createAddAction()
    {
        addAction = new Action()
        {

            @Override
            public void run()
            {
                try {
                    String title = "RSS View - Add new feed";
                    String desc = "Enter the URL of feed to be added.\nIt has to start with the protocoll (e.g. \"http://\")";
                    IInputValidator validator = new IInputValidator()
                    {

                        @Override
                        public String isValid(String newText)
                        {
                            try {
                                new URL(newText);
                            } catch (MalformedURLException ex) {
                                return "Invalid URL";
                            }
                            return null;
                        }

                    };
                    InputDialog dialog = new InputDialog(null, title, desc, "http://", validator);
                    int result = dialog.open();
                    if (result == Window.OK) {

                        String url = dialog.getValue();
                        addFeedUrl(url);
                    }
                } catch (RuntimeException ex) {
                	Logger.error("Could not add new feed.", ex);
                    ExceptionHistory.getInstance().add(ex, "addAction");
                }
            }

        };
        addAction.setText("Add new Feed");
    }

    private void createReloadAction()
    {
        reloadAction = new Action()
        {
            @Override
            public void run()
            {
                if (controller != null) {
                	controller.refreshChannelSettings(new NullProgressMonitor());
                    controller.triggerUpdate();
                }
            }
        };

        reloadAction.setText("Reload Feeds");
        ImageDescriptor desc = ImageBank.getInstance().getImageDescriptor(ImageBank.RELOAD);
        reloadAction.setImageDescriptor(desc);
    }

    private void createOpenFeedAction()
    {
        openFeedAction = new Action()
        {
            @Override
            public void run()
            {
            	 StructuredSelection feed = (StructuredSelection) feedTable.getViewer().getSelection();
            	 ViewItem feedItem = (ViewItem)feed.getFirstElement();
            	 String url = feedItem.getUrl();
                 if (url == null) {
                	 openErrorDialog("RSS feed error", "No link was detected for this feed item");
                     return;
                 }
            	 try {
					controller.getUrlOpener().openUrl(url);
				} catch (Exception e) {
					Logger.error("Could not launch browser for URL '" + url + "'.", e);
                    openErrorDialog("RSS feed error", "Could not launch browser for URL '" + url + "'");
				}
            }
        };

        openFeedAction.setText("Open Feed");
    }

    private void createShowErrorsAction()
    {
        showErrorsAction = new Action()
        {
            @Override
            public void run()
            {
                Shell shell = getViewSite().getShell();
                ExceptionHistory history = ExceptionHistory.getInstance();
                ExceptionHistoryDialog dialog = new ExceptionHistoryDialog(shell, history);
                dialog.open();
            }
        };

        showErrorsAction.setText("Show Error History");
    }

    @Override
    public void setFocus()
    {
        feedTable.getViewer().getControl().setFocus();
    }

    public void setUpdateState(final boolean updating)
    {
        this.updating = updating;
        checkChangeTitleImage();
        boolean changeIcon = preferenceStore.getBoolean(PrefsConstants.CHANGE_ICON_SYNC);
        if (updating && !changeIcon) {
            return;
        }
    }

    /**
     * @param updating
     */
    private void checkChangeTitleImage()
    {
        boolean changeIconPrefs = preferenceStore.getBoolean(PrefsConstants.CHANGE_ICON_SYNC);

        Image image;
        if (changeIconPrefs && updating) {
            image = imageReload;
        } else {
            if (availableItemCount > 0) {
                image = imageTitle;
            } else {
                image = imageTitleNoNews;
            }
        }
        if (currentTitleImage != image) {
            currentTitleImage = image;

            Runnable uiUpdater = new Runnable()
            {
                @Override
                public void run()
                {
                    setTitleImage(currentTitleImage);
                }
            };
            safeAsyncExec(uiUpdater);
        }
    }

    /**
     * @param uiUpdater
     */
    private void safeAsyncExec(Runnable uiUpdater)
    {
        Shell widget = getViewSite().getShell();
        if (widget != null && !widget.isDisposed()) {
            widget.getDisplay().asyncExec(uiUpdater);
        }
    }

    public void openErrorDialog(final String title, final String msg)
    {
        Runnable runDialog = new Runnable()
        {
            @Override
            public void run()
            {
                Shell widget = getViewSite().getShell();
                if (widget == null || widget.isDisposed())
                    return;
                MessageDialog.openError(widget, title, msg);
            }
        };
        safeAsyncExec(runDialog);
    }

    public void setItemCount(int displayedItemCount, int availableItemCount)
    {
        this.availableItemCount = availableItemCount;
        this.displayedItemCount = displayedItemCount;
        Runnable runDialog = new Runnable()
        {
            @Override
            public void run()
            {
                int count = FeedView.this.availableItemCount;
                setPartName("Feeds (" + count + ")");
            }
        };
        safeAsyncExec(runDialog);
        checkChangeTitleImage();
    }

}