// Created on 18.02.2006
package biz.junginger.newsfeed.eclipse;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.graphics.Image;

import biz.junginger.newsfeed.RelevantDateComparator;
import biz.junginger.newsfeed.TemplateUrlListComparator;
import biz.junginger.newsfeed.TooltipTextCreator;
import biz.junginger.newsfeed.model.ChannelCommon;
import biz.junginger.newsfeed.model.ChannelContent;
import biz.junginger.newsfeed.model.ChannelItem;
import biz.junginger.newsfeed.model.ChannelModel;
import biz.junginger.newsfeed.model.ChannelModelListener;
import biz.junginger.newsfeed.util.StringUtils;

/**
 * Transforms the model into actually viewed items according to the current mode
 * (e.g. flat, grouped). On the fly it calls the filter to select displayed
 * items. TODO It also has view state info (e.g. collapsed); could be refactored
 * into a new class.
 */

/**
 * (C) Copyright 2004-2008 Markus Junginger.
 * 
 * @author Markus Junginger
 */
/*
 * This file is part of RSS View.
 * 
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 * 
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class Model2ViewModel implements ChannelModelListener
{
    final static public int MODE_TIME_SORTED = 1;

    final static public int MODE_GROUPED_BY_FEED = 2;

    final static public int MODE_GROUPED_BY_TIME = 3;

    final static public int MODE_GROUPED_BY_FEED_KEEP_SEQ = 4;

    private static final String HTML_PREFIX = "<html>\n<head>\n<style type=\"text/css\">"
            + "body { font-family:Arial,sans-serif; font-size:9pt; font-weight:normal; }\n" + "</style>\n</head><body>";

    private static final String HTML_POSTFIX = "</body></html>";

    private ChannelModel model;

    private FeedView view;

    private TooltipTextCreator tooltipTextCreator;

    private boolean showHeadlineInTooltip;

    private Image newsImage;

    private int imageLevelForGroupedItems;

    private int mode;

    private final IPreferenceStore preferences;

    private ChannelFilter channelFilter;

    private List collapsedKeys;

    private int availableItemCount;

    private int displayedItemCount;

    public Model2ViewModel(ChannelModel model, IPreferenceStore preferenceStore)
    {
        this.model = model;
        this.preferences = preferenceStore;
        model.addChannelModelListener(this);
        tooltipTextCreator = new TooltipTextCreator();
        newsImage = ImageBank.getInstance().createImage(ImageBank.NEWS);
        channelFilter = new ChannelFilter(preferences);
        collapsedKeys = new ArrayList();
    }

    public void setView(FeedView view)
    {
        this.view = view;
        updateView();
    }

    public void dispose()
    {
        if (newsImage != null)
            newsImage.dispose();
        newsImage = null;
    }

    /**
     * Updates the view according to the current model. TODO May be called from
     * different threads currently (counts are not synchronized, e.g.).
     */
    public void updateView()
    {
        displayedItemCount = 0;
        availableItemCount = 0;
        if (view != null) {
            ViewItem[] viewItems = null;
            if (mode == MODE_TIME_SORTED) {
                viewItems = createFlatViewModel();
            } else if (mode == MODE_GROUPED_BY_FEED || mode == MODE_GROUPED_BY_FEED_KEEP_SEQ) {
                boolean keep = mode == MODE_GROUPED_BY_FEED_KEEP_SEQ;
                viewItems = createGroupedByFeedViewModel(keep);
            } else if (mode == MODE_GROUPED_BY_TIME) {
                viewItems = createGroupedByTime();
            } else {
                throw new IllegalArgumentException("Mode: " + mode);
            }
            if (view != null) {
                view.setViewItems(viewItems);
                view.setItemCount(displayedItemCount, availableItemCount);
            }
        }
    }

    private ViewItem[] createGroupedByTime()
    {
        ChannelContent[] channels = model.createChannelArray();
        ChannelItem[] items = flatenAndFilterAndSortByDate(channels);

        List all = new ArrayList();
        List hour = new ArrayList();
        List someHours = new ArrayList();
        List day = new ArrayList();
        List week = new ArrayList();
        List old = new ArrayList();
        List unknown = new ArrayList();

        for (int i = 0; i < items.length; i++) {
            ChannelItem item = items[i];
            Date date = item.getRelevantDate();
            if (date != null) {
                long diff = System.currentTimeMillis() - date.getTime();
                int minutes = (int) ((diff / 1000) / 60);
                if (minutes < 60) {
                    hour.add(item);
                } else if (minutes < 60 * 4) {
                    someHours.add(item);
                } else if (minutes < 60 * 24) {
                    day.add(item);
                } else if (minutes < 60 * 24 * 7) {
                    week.add(item);
                } else {
                    old.add(item);
                }
            } else {
                unknown.add(item);
            }
        }

        merge(all, hour, "Last hour");
        merge(all, someHours, "Last 4 hours");
        merge(all, day, "Last 24 hours");
        merge(all, week, "This week");
        merge(all, old, "Older than a week");
        merge(all, unknown, "Unknown time");

        ViewItem[] itemArray = (ViewItem[]) all.toArray(new ViewItem[0]);
        return itemArray;
    }

    private void merge(List all, List part, String text)
    {
        if (!part.isEmpty()) {
            int size = part.size();
            ViewItem groupViewItem = new ViewItem();
            groupViewItem.setTitle(text + " (" + size + ")");
            groupViewItem.setBoldText(true);
            groupViewItem.setCollapsable(true);
            all.add(groupViewItem);

            boolean collapsed = updateCollapsed(groupViewItem);
            if (!collapsed) {
                Iterator iterator = part.iterator();
                while (iterator.hasNext()) {
                    ChannelCommon item = (ChannelCommon) iterator.next();
                    all.add(createViewItem(item, 1));
                    displayedItemCount++;
                }
            }
        }
    }

    private ViewItem[] createFlatViewModel()
    {
        ChannelContent[] channels = model.createChannelArray();
        ChannelItem[] items = flatenAndFilterAndSortByDate(channels);

        ViewItem[] viewItems = new ViewItem[items.length];
        for (int i = 0; i < viewItems.length; i++) {
            viewItems[i] = createViewItem(items[i], 1);
            displayedItemCount++;
        }
        return viewItems;
    }

    public ChannelItem[] flatenAndFilterAndSortByDate(ChannelContent[] channels)
    {
        List allItems = new ArrayList();
        for (int i = 0; i < channels.length; i++) {
            if (channels[i] != null) {
                List items = channels[i].getChannelItems();
                items = channelFilter.filterChannelItems(items);
                allItems.addAll(items);
            }
        }
        Collections.sort(allItems, new RelevantDateComparator());
        int size = allItems.size();
        availableItemCount += size;
        ChannelItem[] itemArray = (ChannelItem[]) allItems.toArray(new ChannelItem[size]);
        return itemArray;
    }

    private ViewItem[] createGroupedByFeedViewModel(boolean keep)
    {
        ArrayList list = new ArrayList();
        ChannelContent[] channels = model.createChannelArray();
        Comparator comparator;
        if (keep) {
            comparator = new RelevantDateComparator();
        } else {
            comparator = new TemplateUrlListComparator(model.getSettingsCopy());
        }
        Arrays.sort(channels, comparator);
        for (int i = 0; i < channels.length; i++) {
            ChannelContent content = channels[i];
            ViewItem groupViewItem = createViewItem(content, 1);
            groupViewItem.setBoldText(true);
            list.add(groupViewItem);
            groupViewItem.setCollapsable(true);

            List items = content.getChannelItems();
            int sizeUnfiltered = items.size();
            items = channelFilter.filterChannelItems(items);
            int sizeFiltered = items.size();

            String text = groupViewItem.getTitle() + " (";
            if (sizeFiltered == sizeUnfiltered) {
                text += sizeFiltered;
            } else {
                text += sizeFiltered + "/" + sizeUnfiltered;
            }
            text += ")";
            groupViewItem.setTitle(text);
            availableItemCount += sizeFiltered;

            boolean collapsed = updateCollapsed(groupViewItem);
            if (!collapsed) {
                Iterator iter = items.iterator();
                while (iter.hasNext()) {
                    ChannelItem item = (ChannelItem) iter.next();
                    list.add(createViewItem(item, imageLevelForGroupedItems));
                    displayedItemCount++;
                }
            }
        }

        ViewItem[] viewItems = new ViewItem[list.size()];
        viewItems = (ViewItem[]) list.toArray(viewItems);
        return viewItems;
    }

    private boolean updateCollapsed(ViewItem groupViewItem)
    {
        String key = getCollapseKey(groupViewItem);
        boolean collapsed = collapsedKeys.contains(key);
        groupViewItem.setCollapsed(collapsed);
        return collapsed;
    }

    private String getCollapseKey(ViewItem groupViewItem)
    {
        String key = mode + ":" + groupViewItem.getTitle() + ":" + groupViewItem.getUrl();
        return key;
    }

    private ViewItem createViewItem(ChannelCommon item, int imageLevel)
    {
        ViewItem viewItem = new ViewItem();
        viewItem.setObject(item);

        String title = item.getTitle();
        String html = item.getDescription();
        if (title != null) {
            title = StringUtils.convertHtmlToPlainTextNoSpecialChars(title);
            if (title.length() > 80) {
                title = title.substring(0, 80 - 3) + "...";
            }
            viewItem.setTitle(title);
        } else if (html != null) {
            String description = StringUtils.convertHtmlToPlainText(html);
            viewItem.setTitle(description);
        } else {
            viewItem.setTitle("");
        }

        if (imageLevel > 0) {
            Image icon = item.getImage();
            if (icon == null) {
                icon = newsImage;
            }
            if (icon != null) {
                if (imageLevel == 1) {
                    viewItem.setImageLevel1(icon);
                } else if (imageLevel == 2) {
                    viewItem.setImageLevel2(icon);
                } else {
                    throw new IllegalArgumentException("Invalid level: " + imageLevel);
                }
            }
        }
        if (html != null) {
            if (html.indexOf("<body>") == -1 && html.indexOf("<html>") == -1) {
                html = HTML_PREFIX + html + HTML_POSTFIX;
            }
            viewItem.setHtmlContent(html);
        }
        String tooltip = tooltipTextCreator.createPlainText(item, showHeadlineInTooltip);
        viewItem.setPlainTextContent(tooltip);
        DateFormat dateFormat = DateFormat.getInstance();
        Date publishDate = item.getPublishDate();
        if (publishDate != null) {
            viewItem.setDatePublished(dateFormat.format(publishDate));
        } else {
            viewItem.setDatePublished(item.getPublishDateStr());
        }

        Date updatedDate = item.getUpdatedDate();
        if (updatedDate != null) {
            viewItem.setDateUpdated(dateFormat.format(updatedDate));
        } else {
            viewItem.setDateUpdated(item.getUpdatedDateStr());
        }

        viewItem.setUrl(item.getLink());

        return viewItem;
    }

    public void setShowHeadlineInTooltip(boolean showHeadlineInTooltip)
    {
        this.showHeadlineInTooltip = showHeadlineInTooltip;
    }

    public void changedChannelSettings()
    {
        updateView();
    }

    public void updatedChannelContent(ChannelContent content)
    {
        updateView();
    }

    public void setMode(int mode)
    {
        this.mode = mode;
    }

    public void expandOrCollapse(ViewItem item)
    {
        String key = getCollapseKey(item);
        if (item.isCollapsed()) {
            collapsedKeys.remove(key);
        } else {
            collapsedKeys.add(key);
        }
    }

}
