package biz.junginger.newsfeed.eclipse.prefs;

import java.io.IOException;
import java.util.List;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

import biz.junginger.newsfeed.ExceptionHistory;
import biz.junginger.newsfeed.eclipse.FeedPlugin;
import biz.junginger.newsfeed.eclipse.Logger;

/**
 * (C) Copyright 2004-2008 Markus Junginger.
 * 
 * @author Markus Junginger
 */

/*
 * Copyright (c) 2009 Wind River Systems, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Lesser GNU General Public License version
2.1 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the Lesser GNU General Public License for more details.
 *
 * You should have received a copy of the Lesser GNU General Public License version 2.1
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/* Contributor: 
 * Caroline Rieder (Wind River) - removed channel refresh when opening the preference page
 * Caroline Rieder (Wind River) - changed printStackTrace() to logged error
 */

/*
 * This file is part of RSS View.
 * 
 * RSS View is free software: you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation, either version 3 of the License, or (at your option) any
 * later version.
 * 
 * RSS View is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with RSS View. If not, see <http://www.gnu.org/licenses/>.
 */
public class FeedPreferencePage extends PreferencePage implements IWorkbenchPreferencePage
{
    private static final String[] BROWSER_VIEW_ITEMS = {
        "External browser",
        "Single view",
        "Mutiple views",
    };

    private static final String[] BROWSER_PREF_ITEMS = {
        PrefsConstants.VALUE_BROWSER_EXTERNAL,
        PrefsConstants.VALUE_BROWSER_INTERNAL_SINGLE, 
        PrefsConstants.VALUE_BROWSER_INTERNAL_MULTI, 
       };

    private Combo updateIntervalCombo;

    private FeedEditor channelEditor;

    private Button showHeadlineCheckbox;

    private Button changeIconCheckbox;

    private Button availableOfflineCheckbox;

    private Combo browserCombo;

    protected Control createContents(Composite parent)
    {
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 3;
        parent.setLayout(gridLayout);

        createSimpleControls(parent);
        createTableControls(parent);
        createButtons(parent);

        retrieveValues(false);
        return new Composite(parent, SWT.NULL);
    }

    private void createTableControls(Composite parent)
    {
        Label chLabel = new Label(parent, 0);
        chLabel.setText("RSS/Atom Channels: ");
        GridData gdText = new GridData();
        gdText.horizontalSpan = 3;
        chLabel.setLayoutData(gdText);

        channelEditor = new FeedEditor(parent);
        GridData gdTable = new GridData();
        gdTable.horizontalAlignment = GridData.FILL;
        gdTable.grabExcessHorizontalSpace = true;
        gdTable.verticalAlignment = GridData.FILL;
        gdTable.grabExcessVerticalSpace = true;
        gdTable.horizontalSpan = 3;
        channelEditor.setLayoutData(gdTable);
    }

    private void createButtons(Composite parent)
    {
        Composite buttonComposite = new Composite(parent, 0);
        buttonComposite.setLayout(new RowLayout());
        GridData compGd = new GridData();
        compGd.horizontalSpan = 3;
        compGd.horizontalAlignment = SWT.LEFT;
        buttonComposite.setLayoutData(compGd);

        Button addButton = new Button(buttonComposite, SWT.PUSH);
        addButton.setText("Add");
        addButton.addSelectionListener(new SelectionAdapter()
        {
            public void widgetSelected(SelectionEvent e)
            {
                channelEditor.addNewRow();
            }
        });

        Button removeButton = new Button(buttonComposite, SWT.PUSH);
        removeButton.setText("Remove");
        removeButton.addSelectionListener(new SelectionAdapter()
        {
            public void widgetSelected(SelectionEvent e)
            {
                channelEditor.removeCurrentRow();
            }
        });

        Button upButton = new Button(buttonComposite, SWT.PUSH);
        upButton.setText("Up");
        upButton.addSelectionListener(new SelectionAdapter()
        {
            public void widgetSelected(SelectionEvent e)
            {
                channelEditor.moveUpCurrentRow();
            }
        });

        Button downButton = new Button(buttonComposite, SWT.PUSH);
        downButton.setText("Down");
        downButton.addSelectionListener(new SelectionAdapter()
        {
            public void widgetSelected(SelectionEvent e)
            {
                channelEditor.moveDownCurrentRow();
            }
        });

    }

    private void createSimpleControls(Composite parent)
    {
        Label browserLabel = new Label(parent, 0);
        browserLabel.setText("Use browser: ");
        browserCombo = new Combo(parent, SWT.HORIZONTAL | SWT.READ_ONLY);
        browserCombo.setItems(BROWSER_VIEW_ITEMS);
        GridData browserGd = new GridData();
        browserGd.horizontalSpan = 2;
        browserCombo.setLayoutData(browserGd);

        Label updateLabel = new Label(parent, 0);
        updateLabel.setText("Default update interval in minutes: ");
        updateIntervalCombo = new Combo(parent, SWT.HORIZONTAL);
        String delayItems[] = { "5", "10", "15", "30", "60" };
        updateIntervalCombo.setItems(delayItems);
        GridData gd = new GridData();
        gd.horizontalSpan = 2;
        // updateIntervalCombo.setLayoutData(gd);
        new Label(parent, 0);

        showHeadlineCheckbox = new Button(parent, SWT.CHECK);
        showHeadlineCheckbox.setText("Show headline in tooltip");

        changeIconCheckbox = new Button(parent, SWT.CHECK);
        changeIconCheckbox.setText("Change icon when synchronizing");

        new Label(parent, 0);

        availableOfflineCheckbox = new Button(parent, SWT.CHECK);
        availableOfflineCheckbox.setText("Make feeds available offline");

    }

    /** @see org.eclipse.jface.preference.PreferencePage#doGetPreferenceStore() */
    protected IPreferenceStore doGetPreferenceStore()
    {
        return FeedPlugin.getDefault().getPreferenceStore();
    }

    private void retrieveValues(boolean defaults)
    {
        IPreferenceStore store = getPreferenceStore();

        int updateInterval;
        boolean showHeadline;
        boolean changeIcon;
        boolean availableOffline;
        String browser;

        if (defaults) {
            updateInterval = store.getDefaultInt(PrefsConstants.DEFAULT_UPDATE_INTERVAL);
            showHeadline = store.getDefaultBoolean(PrefsConstants.SHOW_HEADLINE_IN_TOOLTIP);
            changeIcon = store.getDefaultBoolean(PrefsConstants.CHANGE_ICON_SYNC);
            availableOffline = store.getDefaultBoolean(PrefsConstants.AVAILABLE_OFFLINE);
            browser = store.getDefaultString(PrefsConstants.BROWSER);
        } else {
            updateInterval = store.getInt(PrefsConstants.DEFAULT_UPDATE_INTERVAL);
            showHeadline = store.getBoolean(PrefsConstants.SHOW_HEADLINE_IN_TOOLTIP);
            changeIcon = store.getBoolean(PrefsConstants.CHANGE_ICON_SYNC);
            availableOffline = store.getBoolean(PrefsConstants.AVAILABLE_OFFLINE);
            browser = store.getString(PrefsConstants.BROWSER);
        }
        if(PrefsConstants.VALUE_BROWSER_CUSTOM.equals(browser)) {
            // Map old value
            browser = PrefsConstants.VALUE_BROWSER_EXTERNAL;
        }
        updateIntervalCombo.setText(Integer.toString(updateInterval));
        showHeadlineCheckbox.setSelection(showHeadline);
        changeIconCheckbox.setSelection(changeIcon);
        availableOfflineCheckbox.setSelection(availableOffline);
        browserCombo.setText(getBrowserViewItem(browser));

        FeedPlugin plugin = FeedPlugin.getDefault();
        List settings = plugin.getModel().getSettingsCopy();
        channelEditor.setChannelSettings(settings);
    }

    private String getBrowserViewItem(String browser)
    {
        for (int i = 0; i < BROWSER_PREF_ITEMS.length; i++) {
            if (BROWSER_PREF_ITEMS[i].equals(browser)) {
                return BROWSER_VIEW_ITEMS[i];
            }
        }
        throw new IllegalArgumentException("Browser pref "+browser);
    }

    private String getBrowserPrefItem(String browser)
    {
        for (int i = 0; i < BROWSER_VIEW_ITEMS.length; i++) {
            if (BROWSER_VIEW_ITEMS[i].equals(browser)) {
                return BROWSER_PREF_ITEMS[i];
            }
        }
        throw new IllegalArgumentException("Browser view "+browser);
    }

    private void storeValues()
    {
        IPreferenceStore store = getPreferenceStore();

        int updateInterval = store.getDefaultInt(PrefsConstants.DEFAULT_UPDATE_INTERVAL);
        try {
            String str = updateIntervalCombo.getText();
            updateInterval = Integer.parseInt(str);
        } catch (Exception ex) {
            Logger.error("RSS-View: Invalid integer value (PREF_DEFAULT_UPDATE_INTERVAL)");
        }
        store.setValue(PrefsConstants.DEFAULT_UPDATE_INTERVAL, updateInterval);

        String browser = getBrowserPrefItem(browserCombo.getText());
        store.setValue(PrefsConstants.BROWSER, browser);
        store.setValue(PrefsConstants.SHOW_HEADLINE_IN_TOOLTIP, showHeadlineCheckbox.getSelection());
        store.setValue(PrefsConstants.CHANGE_ICON_SYNC, changeIconCheckbox.getSelection());
        store.setValue(PrefsConstants.AVAILABLE_OFFLINE, availableOfflineCheckbox.getSelection());

        List settings = channelEditor.getChannelSettings();
        try {
            String settingsStr = FeedSettingsMemento.createXmlMementoString(settings);
            store.setValue(PrefsConstants.RSS_CHANNELS, settingsStr);
        } catch (IOException e) {
        	Logger.error("Could not create XMLMementoString", e); //$NON-NLS-1$
            ExceptionHistory.getInstance().add(e, "FeedPreferencePage");
        }
    }

    /** @see org.eclipse.jface.preference.PreferencePage#performApply() */
    protected void performApply()
    {
        super.performApply();
        storeValues();
    }

    /** @see org.eclipse.jface.preference.IPreferencePage#performOk() */
    public boolean performOk()
    {
        super.performOk();
        storeValues();
        return true;
    }

    protected void performDefaults()
    {
        super.performDefaults();
        retrieveValues(true);
    }

    public void init(IWorkbench workbench)
    {
    }
}